
% ***********************************************************************
% *                           Simulation 2                              *    
% ***********************************************************************

%% ====== Multi-compartment Hodgkin-Huxley Neuron Model Simulation ======
%
% Models action potential propagation along an axon using a spatially 
% discretised Hodgkin-Huxley framework. Includes configurable external 
% stimuli, membrane, and axon parameters. Solves membrane voltage using a 
% finite difference method with a half-step Backward Euler scheme. Outputs 
% voltage propagation plots.

% Author: Oliver Foley
% Date: 29/04/2025

% Parts of code adapted from:
% File Name: stEcab.m
% Book Title: Mathematics for Neuroscientists
% Author: Gabbiani & Cox
% Date: 2017

%% SIMULATION 2: Multi-Compartment Hodgkin-Huxley Axon Model

%% ================= PARAMETERS =================
% Hodgkin-Huxley model parameters for ionic channels and membrane properties
Cm = 1.0;                      % Membrane capacitance (uF/cm^2)
gNa_max = 120;                 % Sodium maximum conductance (mS/cm^2)
gK_max = 36;                   % Potassium maximum conductance (mS/cm^2)
gCl = 0.0667;                  % Chloride leak conductance (mS/cm^2)
ENa = 56;                      % Sodium reversal potential (mV)
EK = -77;                      % Potassium reversal potential (mV)
ECl = -68;                     % Chloride reversal potential (mV)
Ra = 0.3;                      % Axial resistivity (kOhm*cm)
axon_radius = 0.0001;          % Axon radius (cm)
axon_length = 10 / 10;         % Axon length (converted from mm to cm)

%% ================= SIMULATION SETTINGS =================
dx = 0.005;                             % Spatial step size (cm)
dt = 0.05;                              % Temporal step size (ms)
tmax = 50;                              % Maximum simulation time (ms)
t = 0:dt:tmax;                          % Time vector (ms)
Nx = round(axon_length / dx);          % Number of spatial compartments
x = dx/2 : dx : axon_length - dx/2;    % Spatial midpoint vector (cm)
Nt = length(t);                        % Number of time steps

%% ================= STIMULUS SETUP =================
RecLoc1 = 0.25;                         % Recording location 1 (fraction of axon length)
RecLoc2 = 0.75;                         % Recording location 2 (fraction of axon length)

dist1 = RecLoc1 * axon_length * 10;    % Convert recording location 1 to mm
dist2 = RecLoc2 * axon_length * 10;    % Convert recording location 2 to mm

stim_start = 5;                        % Start time of stimulus 1 (ms)
stim_dur = 5;                          % Duration of stimulus 1 (ms)
stim_end = stim_start + stim_dur;      % End time of stimulus 1 (ms)
stim_amp = 1 / 1000;                   % Amplitude of stimulus 1 (uA to mA)

stim2_start = stim_end + 20;           % Start time of stimulus 2 (ms)
stim2_dur = 5;                          % Duration of stimulus 2 (ms)
stim2_end = stim2_start + stim2_dur;   % End time of stimulus 2 (ms)
stim2_amp = 0 / 1000;                  % Amplitude of stimulus 2 (uA to mA)

stim_percentage = 1;                   % Stimulus location as % of axon length
stim_index = round(Nx * (stim_percentage / 100)); % Convert location to index
stim_current = stim_amp / (2 * pi * axon_radius * dx);     % Current density for stimulus 1
stim2_current = stim2_amp / (2 * pi * axon_radius * dx);   % Current density for stimulus 2

%% ================= INITIAL CONDITIONS =================
V = ECl * ones(Nx,1);                  % Initialise membrane potential to resting (mV)
Vm = zeros(Nx,Nt);                     % Preallocate membrane voltage matrix
Vm(:,1) = V;                           % Set initial condition

% Initialise gating variables to steady-state values
n = an(V) ./ (an(V) + bn(V));
m = am(V) ./ (am(V) + bm(V));
h = ah(V) ./ (ah(V) + bh(V));

% Initialise stimulus vectors
I1 = zeros(Nx,1);
I2 = zeros(Nx,1);

% Set up finite difference matrix for spatial coupling
e = ones(Nx,1);
B = spdiags([-e 2*e -e], -1:1, Nx, Nx) / dx^2;
B(1,1) = 1/dx^2; 
B(end,end) = 1/dx^2;
B = (axon_radius / (2 * Ra)) * B;       % Multiply by axial conductance factor
dB = diag(B);                           % Store diagonal for modification each step

%% ================= SOLVE SYSTEM =================
for j = 2:Nt
    t_now = t(j-1);

    % Apply stimulus currents based on time conditions
    I1(stim_index) = stim_current * (t_now > stim_start) * (t_now < stim_end);
    I2(stim_index) = stim2_current * (t_now > stim2_start) * (t_now < stim2_end);

    % Update gating variables using trapezoidal approximation
    a = an(V); c = (a + bn(V))/2;
    n = ((1/dt - c).*n + a) ./ (1/dt + c); n4 = n.^4;

    a = am(V); c = (a + bm(V))/2;
    m = ((1/dt - c).*m + a) ./ (1/dt + c);

    a = ah(V); c = (a + bh(V))/2;
    h = ((1/dt - c).*h + a) ./ (1/dt + c);
    m3h = m.^3 .* h;

    % Update voltage using half-step Backward Euler scheme
    d = gNa_max * m3h + gK_max * n4 + gCl;
    f = gNa_max * m3h * ENa + gK_max * n4 * EK + gCl * ECl + I1 + I2;
    B(1:Nx+1:end) = dB + d + 2*Cm/dt; % Modify diagonal with time and conductance terms
    Vmid = B \ (2*Cm*V/dt + f);      % Solve linear system
    V = 2 * Vmid - V;                % Advance voltage another half step
    Vm(:,j) = V;                     % Store updated voltage profile
end

%% ================= PLOTS =================
% 1. Recording electrode traces overlay
figure(1);
set(gcf, 'Color', 'w');
plot(t, Vm(round(Nx*RecLoc1),:), 'r', t, Vm(round(Nx*RecLoc2),:), 'b', 'LineWidth', 1.5);
title('Action Potential Recordings along the Axon');
xlabel('Time (ms)'); ylabel('Membrane Voltage (mV)');
legend(sprintf('Recording Electrode 1: %.2f mm', dist1), ...
       sprintf('Recording Electrode 2: %.2f mm', dist2));
grid on;

% 2. 2D Heatmap of Voltage Propagation
figure(2);
set(gcf, 'Color', 'w');
imagesc(x*10, t, Vm');
cb = colorbar('Location', 'westoutside'); 
cb.Label.String = 'Membrane Voltage (mV)'; 
cb.Label.FontSize = 12; 
title('2D Analysis of Membrane Voltage Propagation');
xlabel('Axon Length (mm)'); ylabel('Time (ms)');

% 3. 3D Line Plot of Voltage Propagation
figure(3);
set(gcf, 'Color', 'w');
hold on;
for i = 1:round(Nx/50):Nx
    plot3(t, x(i)*10*ones(size(t)), Vm(i,:), 'k');
end
xlabel('Time (ms)'); ylabel('Axon Length (mm)'); zlabel('Membrane Voltage (mV)');
title('Membrane Voltage Propagation through Axon Length and Time');
grid on;
view(-17, 45); % Set 3D view angle 

% 4. Animated Membrane Voltage along Axon
for frame = 1:5:Nt
    figure(4);
    set(gcf, 'Color', 'w');
    plot(x*10, Vm(:,frame), 'r', 'LineWidth', 1.5);
    title(sprintf('Voltage Propagation Along Axon Length Through Time'));
    legend(sprintf('t = %.1f ms', t(frame)));
    xlabel('Axon Length (mm)'); ylabel('Membrane Voltage (mV)');
    ylim([-80 50]); grid on;
    pause(0.05);
end

%% ========================= GATING FUNCTIONS =========================
function val = an(V)
    val = 0.01 * (10 - (V + 71)) ./ (exp(1 - (V + 71) / 10) - 1); % alpha_n
end

function val = bn(V)
    val = 0.125 * exp(-(V + 71) / 80);                             % beta_n
end

function val = am(V)
    val = 0.1 * (25 - (V + 71)) ./ (exp(2.5 - (V + 71) / 10) - 1); % alpha_m
end

function val = bm(V)
    val = 4 * exp(-(V + 71) / 18);                                 % beta_m
end

function val = ah(V)
    val = 0.07 * exp(-(V + 71) / 20);                              % alpha_h
end

function val = bh(V)
    val = 1 ./ (exp(3 - (V + 71) / 10) + 1);                       % beta_h
end